<?php
require_once '../config.php';
verificarLogin();

if ($_SESSION['usuario_tipo'] != 'admin') {
    header("Location: ../index.php");
    exit();
}

$titulo = "Criar Usuário";

// Inicializar segurança
$security = new Security($pdo);

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Verificar CSRF
    if (!$security->verifyCSRFToken($_POST['csrf_token'])) {
        die("Token CSRF inválido!");
    }
    
    $nome = $security->cleanInput($_POST['nome']);
    $email = $security->cleanInput($_POST['email']);
    $login = $security->cleanInput($_POST['login']);
    $senha = $_POST['senha'];
    $tipo = $security->cleanInput($_POST['tipo']);
    
    // Validar dados
    $errors = [];
    
    if (!$security->validateEmail($email)) {
        $errors[] = "Email inválido!";
    }
    
    $password_errors = $security->validatePassword($senha);
    if (!empty($password_errors)) {
        $errors = array_merge($errors, $password_errors);
    }
    
    if (empty($errors)) {
        $senha_hash = password_hash($senha, PASSWORD_DEFAULT);
        
        try {
            $stmt = $pdo->prepare("INSERT INTO usuarios (nome, email, login, senha, tipo) VALUES (?, ?, ?, ?, ?)");
            $stmt->execute([$nome, $email, $login, $senha_hash, $tipo]);
            
            // Registrar na auditoria
            $stmt = $pdo->prepare("INSERT INTO security_logs (usuario_id, acao, ip_address, user_agent) VALUES (?, ?, ?, ?)");
            $stmt->execute([$_SESSION['usuario_id'], 'criar_usuario', $_SERVER['REMOTE_ADDR'], $_SERVER['HTTP_USER_AGENT']]);
            
            header("Location: usuarios.php?sucesso=1");
            exit();
        } catch (PDOException $e) {
            if ($e->getCode() == 23000) {
                $erro = "Email ou login já cadastrado!";
            } else {
                $erro = "Erro ao criar usuário: " . $e->getMessage();
            }
        }
    } else {
        $erro = implode("<br>", $errors);
    }
}

// Gerar token CSRF
$csrf_token = $security->generateCSRFToken();
?>
<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo $titulo; ?> - Gestor DREEYTV</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }
        
        :root {
            --primary: #667eea;
            --secondary: #2d3748;
            --danger: #f56565;
        }
        
        body {
            background: #f5f7fa;
            color: #2d3748;
        }
        
        .admin-container {
            display: flex;
            min-height: 100vh;
        }
        
        .sidebar {
            width: 280px;
            background: var(--secondary);
            color: white;
        }
        
        .sidebar-header {
            padding: 30px 25px;
            background: rgba(0,0,0,0.2);
            border-bottom: 1px solid rgba(255,255,255,0.1);
        }
        
        .sidebar-header h2 {
            font-size: 1.5rem;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .sidebar-menu {
            list-style: none;
            padding: 20px 0;
        }
        
        .sidebar-menu li {
            margin: 5px 15px;
        }
        
        .sidebar-menu a {
            display: flex;
            align-items: center;
            gap: 12px;
            padding: 15px 20px;
            color: rgba(255,255,255,0.8);
            text-decoration: none;
            border-radius: 10px;
            transition: all 0.3s;
        }
        
        .sidebar-menu a:hover {
            background: rgba(255,255,255,0.1);
            color: white;
        }
        
        .sidebar-menu a.active {
            background: var(--primary);
            color: white;
        }
        
        .main-content {
            flex: 1;
            padding: 30px;
        }
        
        .header {
            background: white;
            padding: 25px 30px;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            margin-bottom: 30px;
            display: flex;
            justify-content: space-between;
            align-items: center;
        }
        
        .header h1 {
            font-size: 1.8rem;
            background: linear-gradient(135deg, var(--primary), #5a6fd8);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
        }
        
        .user-info {
            display: flex;
            align-items: center;
            gap: 15px;
        }
        
        .user-avatar {
            width: 45px;
            height: 45px;
            border-radius: 50%;
            background: linear-gradient(135deg, var(--primary), #5a6fd8);
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-weight: bold;
        }
        
        .content {
            background: white;
            padding: 30px;
            border-radius: 15px;
            box-shadow: 0 5px 15px rgba(0,0,0,0.1);
            max-width: 600px;
            margin: 0 auto;
        }
        
        .form-header {
            text-align: center;
            margin-bottom: 30px;
        }
        
        .form-header .icon {
            font-size: 3rem;
            margin-bottom: 10px;
        }
        
        .form-group {
            margin-bottom: 20px;
        }
        
        .form-label {
            display: block;
            margin-bottom: 8px;
            font-weight: 600;
            color: #2d3748;
        }
        
        .form-control {
            width: 100%;
            padding: 15px;
            border: 2px solid #e2e8f0;
            border-radius: 10px;
            font-size: 16px;
            transition: all 0.3s;
        }
        
        .form-control:focus {
            border-color: var(--primary);
            outline: none;
            box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
        }
        
        .btn {
            display: inline-flex;
            align-items: center;
            gap: 8px;
            padding: 15px 25px;
            background: var(--primary);
            color: white;
            text-decoration: none;
            border-radius: 10px;
            font-weight: 600;
            transition: all 0.3s;
            border: none;
            cursor: pointer;
            font-size: 16px;
        }
        
        .btn:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.3);
        }
        
        .btn-block {
            display: block;
            width: 100%;
        }
        
        .btn-danger {
            background: var(--danger);
        }
        
        .btn-danger:hover {
            box-shadow: 0 5px 15px rgba(245, 101, 101, 0.3);
        }
        
        .form-actions {
            display: flex;
            gap: 15px;
            margin-top: 30px;
        }
        
        .alert {
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 20px;
        }
        
        .alert-danger {
            background: #fed7d7;
            color: #742a2a;
            border: 1px solid #feb2b2;
        }
        
        .password-info {
            font-size: 0.8rem;
            color: #666;
            margin-top: 5px;
        }
        
        .password-requirements {
            background: #f7fafc;
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 15px;
            font-size: 0.8rem;
        }
        
        .requirement {
            display: flex;
            align-items: center;
            gap: 8px;
            margin-bottom: 5px;
        }
        
        .requirement.valid {
            color: #22543d;
        }
        
        .requirement.invalid {
            color: #742a2a;
        }
        
        @media (max-width: 768px) {
            .admin-container {
                flex-direction: column;
            }
            
            .sidebar {
                width: 100%;
            }
            
            .form-actions {
                flex-direction: column;
            }
        }
    </style>
</head>
<body>
    <div class="admin-container">
        <div class="sidebar">
            <div class="sidebar-header">
                <h2>📺 GESTOR DREEYTV</h2>
            </div>
            <ul class="sidebar-menu">
                <li><a href="../index.php">📊 Dashboard</a></li>
                <li><a href="../clientes/clientes.php">👨‍💼 Clientes</a></li>
                <li><a href="../planos/planos.php">📋 Planos</a></li>
                <li><a href="usuarios.php">👥 Usuários</a></li>
                <li><a href="../senha/alterar_senha.php">🔑 Alterar Senha</a></li>
                <li><a href="../logout.php">🚪 Sair</a></li>
            </ul>
        </div>
        
        <div class="main-content">
            <div class="header">
                <h1>Criar Usuário</h1>
                <div class="user-info">
                    <div class="user-avatar"><?php echo substr($_SESSION['usuario_nome'], 0, 1); ?></div>
                    <div>
                        <div style="font-weight: 600;"><?php echo $_SESSION['usuario_nome']; ?></div>
                        <div style="font-size: 0.9rem; color: #666;"><?php echo ucfirst($_SESSION['usuario_tipo']); ?></div>
                    </div>
                    <a href="../logout.php" class="btn" style="background: var(--danger);">🚪 Sair</a>
                </div>
            </div>
            
            <div class="content">
                <div class="form-header">
                    <div class="icon">👥</div>
                    <h2>Novo Usuário</h2>
                    <p style="color: #666;">Cadastre um novo usuário no sistema</p>
                </div>
                
                <?php if (isset($erro)): ?>
                    <div class="alert alert-danger">
                        <strong>❌ Erro:</strong> <?php echo $erro; ?>
                    </div>
                <?php endif; ?>
                
                <div class="password-requirements">
                    <strong>🔒 Requisitos de Senha:</strong>
                    <div class="requirement invalid" id="req-length">✓ Pelo menos 8 caracteres</div>
                    <div class="requirement invalid" id="req-uppercase">✓ Pelo menos uma letra maiúscula</div>
                    <div class="requirement invalid" id="req-lowercase">✓ Pelo menos uma letra minúscula</div>
                    <div class="requirement invalid" id="req-number">✓ Pelo menos um número</div>
                    <div class="requirement invalid" id="req-special">✓ Pelo menos um caractere especial</div>
                </div>
                
                <form method="POST" id="userForm">
                    <input type="hidden" name="csrf_token" value="<?php echo $csrf_token; ?>">
                    
                    <div class="form-group">
                        <label class="form-label" for="nome">👤 Nome Completo</label>
                        <input type="text" id="nome" name="nome" class="form-control" placeholder="Digite o nome completo" required
                               value="<?php echo isset($_POST['nome']) ? htmlspecialchars($_POST['nome']) : ''; ?>">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="email">📧 Email</label>
                        <input type="email" id="email" name="email" class="form-control" placeholder="exemplo@email.com" required
                               value="<?php echo isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''; ?>">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="login">🔑 Login</label>
                        <input type="text" id="login" name="login" class="form-control" placeholder="Digite o nome de usuário" required
                               value="<?php echo isset($_POST['login']) ? htmlspecialchars($_POST['login']) : ''; ?>">
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="senha">🔒 Senha</label>
                        <input type="password" id="senha" name="senha" class="form-control" placeholder="Digite a senha" required>
                        <div class="password-info">A senha deve atender a todos os requisitos acima</div>
                    </div>
                    
                    <div class="form-group">
                        <label class="form-label" for="tipo">🎯 Tipo de Usuário</label>
                        <select id="tipo" name="tipo" class="form-control" required>
                            <option value="usuario">👤 Usuário</option>
                            <option value="admin">👑 Administrador</option>
                        </select>
                    </div>
                    
                    <div class="form-actions">
                        <button type="submit" class="btn btn-block" id="submitBtn">✅ Cadastrar Usuário</button>
                        <a href="usuarios.php" class="btn btn-danger btn-block">❌ Cancelar</a>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <script>
        document.getElementById('senha').addEventListener('input', function(e) {
            const password = e.target.value;
            
            // Verificar requisitos
            const hasLength = password.length >= 8;
            const hasUppercase = /[A-Z]/.test(password);
            const hasLowercase = /[a-z]/.test(password);
            const hasNumber = /[0-9]/.test(password);
            const hasSpecial = /[^A-Za-z0-9]/.test(password);
            
            // Atualizar visual dos requisitos
            document.getElementById('req-length').className = hasLength ? 'requirement valid' : 'requirement invalid';
            document.getElementById('req-uppercase').className = hasUppercase ? 'requirement valid' : 'requirement invalid';
            document.getElementById('req-lowercase').className = hasLowercase ? 'requirement valid' : 'requirement invalid';
            document.getElementById('req-number').className = hasNumber ? 'requirement valid' : 'requirement invalid';
            document.getElementById('req-special').className = hasSpecial ? 'requirement valid' : 'requirement invalid';
            
            // Habilitar/desabilitar botão
            const allValid = hasLength && hasUppercase && hasLowercase && hasNumber && hasSpecial;
            document.getElementById('submitBtn').disabled = !allValid;
        });

        document.getElementById('userForm').addEventListener('submit', function(e) {
            const password = document.getElementById('senha').value;
            const hasLength = password.length >= 8;
            const hasUppercase = /[A-Z]/.test(password);
            const hasLowercase = /[a-z]/.test(password);
            const hasNumber = /[0-9]/.test(password);
            const hasSpecial = /[^A-Za-z0-9]/.test(password);
            
            if (!(hasLength && hasUppercase && hasLowercase && hasNumber && hasSpecial)) {
                e.preventDefault();
                alert('Por favor, atenda a todos os requisitos de senha!');
            }
        });
    </script>
</body>
</html>